//package globalExamples;

import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.DynamicDefinition;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.SourceGenerator;
import oracle.olapi.data.source.Template;
import oracle.olapi.transaction.metadataStateManager.MetadataState;

/**
 * Creates a TopBottomTemplateState, a TopBottomTemplateGenerator,
 * and a DynamicDefinition.
 * Gets the current state of the TopBottomTemplateState and the values that
 * it stores.
 * Sets the data values stored by the TopBottomTemplateState and sets the 
 * changed state as the current state.
 *
 * <P>
 * The TopBottomTemplateState and TopBottomTemplateGenerator classes are
 * inner classes of this class.
 *
 * <P>
 * This class includes the code for Example 11-1, Implementing a Template, 
 * Example 11-2, Implementing a MetadataState, and Example 11-3, Implementing 
 * a SourceGenerator, in the Chapter 11, Creating Dynamic Queries, in the 
 * Oracle OLAP Developer's Guide to the OLAP API.
 * 
 * @author Oracle Corporation
 */
public class TopBottomTemplate extends Template 
{
  // This outer class is Example 11-1, Implementing a Template.
  /**
   * Specifies selecting the elements from the beginning of the result set.
   */
  public static final int TOP_BOTTOM_TYPE_TOP = 0;
  /**
   * Specifies selecting the elements from the end of the result set.
   */
  public static final int TOP_BOTTOM_TYPE_BOTTOM = 1;

  // Variable to store the DynamicDefinition.
  private DynamicDefinition _definition;

  /**
   * Creates a TopBottomTemplate with a default type and number values
   * and the specified base dimension.
   */
  public TopBottomTemplate(Source base, DataProvider dataProvider) 
  {
    super(new TopBottomTemplateState(base, TOP_BOTTOM_TYPE_TOP, 0),
                                     dataProvider);
    // Create the DynamicDefinition for this Template. Create the 
    // TopBottomTemplateGenerator that the DynamicDefinition uses.
    _definition = 
    createDynamicDefinition(new TopBottomTemplateGenerator(dataProvider));
  }

  /**
   * Gets the Source produced by the TopBottomTemplateGenerator
   * from the DynamicDefinition.
   */
  public final Source getSource() 
  {
    return _definition.getSource();
  }

  /**
   * Gets the Source that is the base of the elements in the result set.
   * Returns null if the state has no base.
   */
  public Source getBase() 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    return state.base;
  }

  /**
   * Sets a Source as the base.
   */
  public void setBase(Source base) 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    state.base = base;
    setCurrentState(state);
  }

  /**
   * Gets the Source that specifies the measure and the single
   * selections from the dimensions other than the base.
   */
  public Source getCriterion() 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    return state.criterion;
  }

  /**
   * Specifies a Source that defines the measure and the single values 
   * selected from the dimensions other than the base.
   * The SingleSelectionTemplate produces such a Source.
   */
  public void setCriterion(Source criterion) 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    state.criterion = criterion;
    setCurrentState(state);
  }

  /**
   * Gets the type, which is either TOP_BOTTOM_TYPE_TOP or 
   * TOP_BOTTOM_TYPE_BOTTOM.
   */
  public int getTopBottomType() 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    return state.topBottomType;
  }

  /**
   * Sets the type.
   */
  public void setTopBottomType(int topBottomType) 
  {
    if ((topBottomType < TOP_BOTTOM_TYPE_TOP) ||
        (topBottomType > TOP_BOTTOM_TYPE_BOTTOM))
      throw new IllegalArgumentException("InvalidTopBottomType");
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    state.topBottomType = topBottomType;
    setCurrentState(state);
  }

  /**
   * Gets the number of elements selected.
   */
  public float getN() 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    return state.N;
  }

  /**
   * Sets the number of elements to select.
   */
  public void setN(float N) 
  {
    TopBottomTemplateState state = (TopBottomTemplateState) getCurrentState();
    state.N = N;
    setCurrentState(state);
  }

  // The TopBottomTemplateState inner class is Example 11-2, Implementing 
  // a MetadataState.
  /**
   * Stores data that can be changed by its TopBottomTemplate.
   * The data is used by a TopBottomTemplateGenerator in producing
   * a Source for the TopBottomTemplate.
   */
   private static final class TopBottomTemplateState
                 implements Cloneable, MetadataState 
  {

    public int topBottomType;
    public float N;
    public Source criterion;
    public Source base;

    /**
     * Creates a TopBottomTemplateState.
     */
    private TopBottomTemplateState(Source base, int topBottomType, float N) 
    {
      this.base = base;
      this.topBottomType = topBottomType;
      this.N = N;
    }

    /**
     * Creates a copy of this TopBottomTemplateState.
     */
    public final Object clone() 
    {
      try 
      {
        return super.clone();
      }
      catch(CloneNotSupportedException e) 
      {
        return null;
      }
    }
  }

  // The TopBottomTemplateGenerator inner class is Example 11-3, 
  // Implementing a SourceGenerator.
  /**
   * Produces a Source for a TopBottomTemplate based on the data
   * values of a TopBottomTemplateState.
   */
  private final class TopBottomTemplateGenerator
      implements SourceGenerator 
  {
    // Store the DataProvider.
    private DataProvider _dataProvider;

    /**
     * Creates a TopBottomTemplateGenerator.
     */
    public TopBottomTemplateGenerator(DataProvider dataProvider) 
    {
      _dataProvider = dataProvider;
    }

    /**
     * Generates a Source for a TopBottomTemplate using the current
     * state of the data values stored by the TopBottomTemplateState.
     */
    public Source generateSource(MetadataState state)
    {
      // Get the current state of the <CODE>MetadataState</CODE> 
      // implementation for the <CODE>Template</CODE>.
      TopBottomTemplateState castState = (TopBottomTemplateState) state;

      if (castState.criterion == null)
        throw new NullPointerException("CriterionParameterMissing");
      
      Source sortedBase = null;

      // Depending on the topBottomType value, select from the base Source 
      // the elements specified by the criterion Source and sort the
      // elements in ascending or descending order. 
      if (castState.topBottomType == TOP_BOTTOM_TYPE_TOP)
        sortedBase = castState.base.sortDescending(castState.criterion);
      else 
        sortedBase = castState.base.sortAscending(castState.criterion);

      // Select the first N number of the sorted elements and return that
      // selection. 
      return sortedBase.interval(1, Math.round(castState.N));
    }
  }
}

 
